<?php

namespace Onestartup\Shop\Controller;

use Illuminate\Http\Request;
use App\Http\Requests;
use App\Http\Controllers\Controller;
use Yajra\Datatables\Datatables;
use Onestartup\Shop\Model\ProductCategoryShop as Category;
use Onestartup\Shop\Model\ProductImageShop as Gallery;
use Onestartup\Shop\Model\ProductShop as Product;
use Onestartup\Shop\Model\ExtraField as Extra;
use Onestartup\Shop\Model\ProductShopInfo as ProductInfo;
use Onestartup\Shop\Model\CategoriaSat;
use Onestartup\Shop\Model\Billing;
use Onestartup\Shop\Model\VariableExtra as Variable;

use Onestartup\Shop\DataTables\ClientDataTable;
use Onestartup\Shop\DataTables\OrderDataTable;
use Onestartup\Shop\DataTables\StockDataTable;

use Onestartup\Shop\Model\SaleShop as Sale;
use Onestartup\Shop\Model\OrderTracking as Tracking;
use Onestartup\Shop\Model\ConfiguracionFactura as Config;
use Onestartup\Shop\Libs\Util;

use Onestartup\Shop\Requests\RequestProduct;
use Onestartup\Shop\Requests\RequestProductInfo;
use Onestartup\Shop\Payment\MP;

use PayPal\Api\Amount;
use PayPal\Api\Refund;
use PayPal\Api\Sale as SalePaypal;
use PayPal\Rest\ApiContext;
use PayPal\Auth\OAuthTokenCredential;
use PayPal\Api\Payer;
use PayPal\Api\Payment;

class AdminProductController extends Controller
{
    protected $util;

    public function __construct()
    {
        $this->util = new Util();
    }

    public function index()
    {
        return view('shop::product.index');
    }

    /**
     * Show the form for creating a new resource.
     * @return Response
     */
    public function create()
    {
        $categories = Category::pluck('name', 'id');
        $categoria_sat = CategoriaSat::pluck('nombre', 'id');
        $extras = Extra::where('active', true)->orderBy('variable', 'asc')->get();

        return view('shop::product.create')
        ->with('categories', $categories)
        ->with('categoria_sat', $categoria_sat)
        ->with('extras', $extras);
    }

    /**
     * Store a newly created resource in storage.
     * @param  Request $request
     * @return Response
     */
    public function store(RequestProduct $request)
    {

        $product = new Product($request->all());

        if (isset($request->cover)) {

            $file = $request->file('cover');
            $nombre = $file->getClientOriginalName();
            $nombre_file = str_replace(' ', '_', $nombre);
            $ubicacion_donde_guarda ='shop/products/cover/'.$nombre_file;
            \Storage::disk('local')->put($ubicacion_donde_guarda,  \File::get($file));
            $product->cover = $ubicacion_donde_guarda;
        } 


        \Auth::user()->productsShop()->save($product);
        

        return redirect()
        ->route('admin.shop.product.edit', $product->id)
        ->with('message_success', "Producto guardado correctamente, continua agregando las imagenes correspondientes");
    }

    /**
     * Show the form for editing the specified resource.
     * @return Response
     */
    public function edit($id)
    {
        $product = Product::find($id);
        $categories = Category::pluck('name', 'id');
        $categoria_sat = CategoriaSat::pluck('nombre', 'id');
        $extras = Extra::where('active', true)->orderBy('variable', 'asc')->get();

        if ($product->infoSale != null) {
            $info = $product->infoSale;
        } else {
            $info = new ProductInfo();
        }

        //return $info;


        return view('shop::product.edit')
        ->with('categories', $categories)
        ->with('categoria_sat', $categoria_sat)
        ->with('product', $product)
        ->with('info', $info)
        ->with('extras', $extras);
    }

    /**
     * Update the specified resource in storage.
     * @param  Request $request
     * @return Response
     */
    public function update(RequestProduct $request, $id)
    {
        $product = Product::find($id);
        $product->fill($request->all());

        if (isset($request->cover)) {

            $file = $request->file('cover');
            $nombre = $file->getClientOriginalName();
            $nombre_file = str_replace(' ', '_', $nombre);
            $ubicacion_donde_guarda ='shop/products/cover/'.$nombre_file;
            \Storage::disk('local')->put($ubicacion_donde_guarda,  \File::get($file));
            $product->cover = $ubicacion_donde_guarda;
        } 


        $product->save();

        return redirect()
        ->back()
        ->with('message_info', 'Información actualizada correctamente');
    }

    /**
     * Remove the specified resource from storage.
     * @return Response
     */
    public function destroy($id)
    {
        $product = Product::find($id);
        $product->delete();

        return redirect()
        ->back()
        ->with('message_danger', 'Producto eliminada correctamente');
    }

    public function getProducts()
    {

        $products = Product::select([
            'id',
            'name',
            'slug', 
            'category_id', 
            'created_at', 
            'description', 
            'features', 
            'specifications'
        ]);

        return Datatables::of($products)
        ->addColumn('main', function ($product) {
            return 
            "<p class='mb0'>Nombre: <b>$product->name</b></p>".
            "<p class='mb0'>Slug: <b>$product->slug</b></p>".
            "<p class='mb0'>Categoria: <b>".$product->category->name."</b></p>".
            "<p class='mb0'>Fecha creación: <b>".$product->created_at->format('d/m/Y')."</b></p>";
        })
        ->addColumn('form', function ($product) {
            return "<center><form method='POST' action='".route('admin.shop.product.destroy',$product->id)."'>".
            "<input name='_method' type='hidden' value='DELETE' class='has-value'>".
            csrf_field() .
            "<button class='btn btn-danger btn-xs button-mb' onclick='return confirm();' type='submit'>".
            "<i class='fas fa-trash-alt icon-special-size'></i>Eliminar</button>".
            "</form>".
            "<a href='".route('show.shop', $product->slug)."' class='btn btn-xs info button-mb' target='new'>".
            "<i class='fas fa-eye icon-special-size'></i>Ver</a>".
            "<br><a href='".route('admin.shop.product.edit', $product->id)."' class='btn btn-xs accent mb0'>".
            "<i class='fas fa-edit icon-special-size'></i>Editar</a></center>";
        })
        ->rawColumns(['main', 'description', 'features', 'specifications', 'form'])
        ->make();
        
    }

    public function storeGallery(Request $request, $product_id)
    {
        $product = Product::find($product_id);

        $asset = '/storage/shop/product/'.$product->id.'/gallery/';
        $path = public_path().$asset;
        $files = $request->file('file');

        foreach($files as $file){
            $fileName = $file->getClientOriginalName();
            $nombre_file = str_replace(' ', '_', $fileName);
            $file->move($path, $nombre_file);
            $gallery = new Gallery([
                'path'=>$asset.$nombre_file
            ]);

            $product->images()->save($gallery);
        }


        return $product;
    }

    public function deleteImage($id)
    {
        $gallery = Gallery::find($id);
        if (file_exists(public_path().$gallery->path)) {
            unlink(public_path().$gallery->path);
        }
        
        $gallery->delete();

        return redirect()
        ->back()
        ->with('message_danger', 'Imagen eliminada correctamente');

    }

    public function deleteCover($id)
    {
        $product = Product::find($id);
        if (file_exists(public_path().'/storage/'.$product->cover)) {
            unlink(public_path().'/storage/'.$product->cover);
        }
        
        $product->cover = null;
        $product->save();

        
        //$gallery->delete();

        return redirect()
        ->back()
        ->with('message_danger', 'Imagen eliminada correctamente');

    }



    public function extraInfo(RequestProductInfo $request)
    {

        $info = ProductInfo::find($request->product_id);

        if ($info == null) {
            $info = new ProductInfo();
        }

        $info->fill($request->all());

        $info->save();

        return redirect()
        ->back()
        ->with('message_success', 'Información de venta actualizada');

    }

    public function listClient(ClientDataTable $dataTable)
    {
        return $dataTable->render('shop::clients.list');
    }

    public function listOrders(OrderDataTable $dataTable)
    {
        return $dataTable->render('shop::clients.orders');
    }

    public function stock(StockDataTable $dataTable)
    {
        return $dataTable->render('shop::product.stock');
    }

    public function showOrder($id)
    {
        $sale = Sale::find($id);
        $promo = null;
        $shipping = null;
        $shipping_price = null;
        $var = Variable::first();
        $variable = $var != null ? $var : new Variable();


        if ($sale->coupon != null) {
            $promo = $this->util->getDiscount($sale, $sale->coupon);
        }
        $client = $sale->client;
        $orden = $sale->items;
        if ($client->shipping != null) {
            $shipping = $client->shipping;
            $shipping_price = $shipping->shipping_price;
        }
        

        if ($sale->status == 0) {
            $estado = "Cancelado";
        }

        if ($sale->status == 1) {
            $estado = "Pendiente";
        }

        if ($sale->status == 2) {
            $estado = "Pagado";
        }

        if ($sale->status == 3) {
            $estado = "En proceso";
        }

        if ($sale->status == 4) {
            $estado = "Cancelado/Reembolsado";
        }

        return view('shop::clients.orden')
            ->with('sale', $sale)
            ->with('client', $client)
            ->with('orden', $orden)
            ->with('estado', $estado)
            ->with('promo', $promo)
            ->with('shipping_price', $shipping_price)
             ->with('variable', $variable)
            ->with('shipping', $shipping);
    }

    public function updateShipping(Request $request, $id)
    {
        $sale = Sale::find($id);
        $sale->shipping_status = $request->shipping_status;
        $sale->save();

        return redirect()->back()->with('message_success', 'Estatus actualizado correctamente');
    }

    public function updateStatusSale(Request $request, $id)
    {
        $sale = Sale::find($id);
        $old = $sale->status;
        $sale->status = $request->status;
        $sale->save();

        $this->util->stock_change($sale, $old);

        return redirect()->back()->with('message_success', 'Estatus actualizado correctamente');
    }

    public function trackingStore(Request $request, $id)
    {
        $sale = Sale::find($id);
        $comment = new Tracking($request->all());
        $comment->user_id = \Auth::user()->id;

        $sale->tracking_sales()->save($comment);

        return redirect()
            ->back()
            ->with('message_success', 'Información agregada correctamente');
    }

    /**
     * Reembolsa pagos hechos con tarjeta
     * @param  [type] $sale_id [description]
     * @return [type]          [description]
     */
    public function refund($sale_id)
    {
        //$paymentInfo = $mp->get_payment ("3799582572");
        //$result = $mp->refund_payment("3799582572");
        //$paymentInfo = $mp->get_payment ($sale->transaction_id);
        $sale = Sale::find($sale_id);
        $mp = new MP(env('AC_MERCADO_PAGO'));
        $result = $mp->refund_payment($sale->transaction_id);
        $old = $sale->status;
        $sale->status = 4;
        $sale->save();

        $this->util->stock_change($sale, $old);


        dd($result);

        return redirect()->back()->with('message_success', 'Se realizo el reembolso correctamente');
    }

    public function refundPaypal($sale_id)
    {
        $saleLocal = Sale::find($sale_id);

        $apiContext = new ApiContext(
            new OAuthTokenCredential(
                env('CLIENT_ID_PAYPAL'),    
                env('SECRET_PAYPAL')
            )
        );


        $apiContext->setConfig(
          array(
            'log.LogEnabled' => true,
            'log.FileName' => base_path().'/Paypal.log',
            'log.LogLevel' => 'DEBUG',
            'mode' => env('PAYPAL_MODE'),
        )
      );

        $payment = Payment::get($saleLocal->transaction_id, $apiContext);

        //dd($payment->transactions[0]->related_resources[0]->sale->id);

        $amt = new Amount();
        $amt->setTotal($saleLocal->total)
          ->setCurrency('MXN');

        $refund = new Refund();
        $refund->setAmount($amt);

        $sale = new SalePaypal();
        $sale->setId($payment->transactions[0]->related_resources[0]->sale->id);

        try {

            $refundedSale = $sale->refund($refund, $apiContext);
            $old = $saleLocal->status;
            $saleLocal->status = 4;
            $saleLocal->save();

            $this->util->stock_change($saleLocal, $old);

          return redirect()->back()->with('message_success', 'Se realizo el reembolso correctamente');
          //dd($refundedSale);

        } catch (PayPal\Exception\PayPalConnectionException $ex) {
          echo $ex->getCode();
          echo $ex->getData();
          die($ex);
        } catch (Exception $ex) {
          die($ex);
        }
        
    }

    public function facturar(Request $request, $sale_id)
    {
        $sale = Sale::find($sale_id);
        $billing = $sale->billing;
        $products = $sale->products;
        $conceptos = [];
        $util = new Util();
        $config = Config::first();

        $emisor = \FacturaDigital::emisor()
            ->setRegimenFiscal($config->regimen_fiscal)
            ->getData();
                            
        $receptor = \FacturaDigital::receptor()
        ->setRfc($billing->rfc)
        ->setNombre($billing->razon)
        ->setUsoCFDI($billing->uso_cfdi)
        ->setCalle($billing->calle != null ? $billing->calle : '')
        ->setNoExt($billing->numero != null ? $billing->numero : '')
        ->setColonia($billing->colonia != null ? $billing->colonia : '')
        ->setMunicipio($billing->ciudad != null ? $billing->ciudad : '')
        ->setEstado($billing->estado != null ? $billing->estado : '')
        ->setPais('Mexico')
        ->setCodigoPostal($billing->cp != null ? $billing->cp : '')
        ->getData();
        

        foreach ($products as $product) {
            $concepto = \FacturaDigital::concepto()
                ->setClaveProdServ($product->categoria_sat->clave_producto)
                ->setNoIdentificacion($product->id)
                ->setCantidad($product->pivot->quantity)
                ->setClaveUnidad($product->categoria_sat->clave_unidad)
                ->setUnidad($product->categoria_sat->unidad)
                ->setDescripcion($product->name)
                ->setValorUnitario($product->infoSale->sale_price)
                ->getData();
            array_push($conceptos, $concepto);
        }

        if ($sale->home_delivery) {
            $envio = $sale->client->shipping->shipping_price;

            $concepto = \FacturaDigital::concepto()
                ->setClaveProdServ('78102203')
                ->setNoIdentificacion(1000)
                ->setCantidad(1)
                ->setClaveUnidad('E48')
                ->setUnidad('Unidad de Servicio')
                ->setDescripcion($envio->name)
                ->setValorUnitario($envio->cost)
                ->getData();
            array_push($conceptos, $concepto);

            
        }

        $lista = \FacturaDigital::listaConceptos();
        $lista->setConceptos($conceptos);
        $totales = $lista->getTotalConceptos();
        $descuento = null;
        $descuento_aux = 0;

        if($sale->coupon != null) {
            $data_discount = $util->getDiscount($sale, $sale->coupon);
            $descuento = $data_discount['amount_unformated'];
            $descuento_aux = $data_discount['amount_unformated'];
        }
       

        $factura = \FacturaDigital::setSerie('F')
        ->setFolio($sale->id)
        ->setFecha('AUTO')
        ->setFormaPago($billing->forma_pago)
        ->setCondicionesDePago('Pago de contado')
        ->setMoneda('MXN')
        ->setTipoCambio('1')
        ->setTipoDeComprobante('I')
        ->setMetodoPago('PUE')
        ->setLugarExpedicion($config->lugar_expedicion) //C.P.
        ->setLeyendaFolio('Folio')
        ->setSubTotal($totales['totalConceptos'])
        ->setDescuento($descuento)
        ->setTotal($totales['totalConceptos'] + $totales['totalImpuestos'] - $descuento_aux );

    
        $impuesto = \FacturaDigital::impuesto()
            ->getData($totales['totalImpuestos']);


        $factura->setEmisor($emisor);
        $factura->setReceptor($receptor);
        $factura->setConceptos($lista->getData());
        $factura->setImpuestos($impuesto);

        //return $factura->getData();

    
        $factura_final = $factura->enviar();

        if ($factura_final['codigo'] == 200) {

            $billing->uuid=$factura_final['cfdi']['UUID'];
            $billing->pdf=$factura_final['cfdi']['PDF'];
            $billing->xml=$factura_final['cfdi']['XML'];
            $billing->status = 1;
            $billing->save();

            $enviar = $factura->enviarCorreo($factura_final['cfdi']['UUID'], $billing->correo, '');

            if ($enviar) {
                return redirect()->back()->with('message_success', "Factura enviada por correo");
            } else {
                return redirect()->back()->with('message_warning', "No se pudo enviar la factura");
            }
        } else {
            return redirect()->back()->with('message_danger', $factura_final['mensaje'].' ó Verifica que los datos del cliente sean correctos');
        }

    }

    public function facturaEnviar(Request $request, $billing_id) {

        $billing = Billing::find($billing_id);

        $enviar = \FacturaDigital::enviarCorreo($billing->uuid, $billing->correo, '');

        if ($enviar) {
                return redirect()->back()->with('message_success', "Factura enviada por correo");
            } else {
                return redirect()->back()->with('message_warning', "No se pudo enviar la factura");
        }
    }
    public function facturaCancelar(Request $request, $billing_id) {
        $billing = Billing::find($billing_id);

        $cancela = \FacturaDigital::cancelarCFDI($billing->uuid);

        if ($cancela['codigo'] != 200) {
            $billing->status = 2;
            $billing->save();
           return redirect()->back()->with('message_warning', $cancela['mensaje']);
        } else {
            return redirect()->back()->with('message_danger', $cancela['mensaje']);
        }
        
    }

    public function showConfigs() {

        $config = Config::first();
        $regimen = [
            '603' => 'Personas Morales con Fines no Lucrativos',
            '605' => 'Sueldos y Salarios e Ingresos Asimilados a Salarios',
            '606' => 'Arrendamiento',
            '608' => 'Demás ingresos',
            '609' => 'Consolidación',
            '610' => 'Residentes en el Extranjero sin Establecimiento Permanente en México',
            '611' => 'Ingresos por Dividendos (socios y accionistas)',
            '612' => 'Personas Físicas con Actividades Empresariales y Profesionales',
            '614' => 'Ingresos por intereses',
            '616' => 'Sin obligaciones fiscales',
            '620' => 'Sociedades Cooperativas de Producción que optan por diferir sus ingresos',
            '621' => 'Incorporación Fiscal',
            '622' => 'Actividades Agrícolas, Ganaderas, Silvícolas y Pesqueras',
            '623' => 'Opcional para Grupos de Sociedades',
            '624' => 'Coordinados',
            '628' => 'Hidrocarburos',
            '607' => 'Régimen de Enajenación o Adquisición de Bienes',
            '629' => 'De los Regímenes Fiscales Preferentes y de las Empresas Multinacionales',
            '630' => 'Enajenación de acciones en bolsa de valores',
            '615' => 'Régimen de los ingresos por obtención de premios'
        ];

        if ($config == null) {
            $config = new Config();
        }

        return view('shop::config_factura.edit')
            ->with('config', $config)
            ->with('regimen', $regimen);
    }

    public function postConfigs(Request $request) {

        $config = Config::first();

        if ($config == null) {
            $config = new Config();
        }

        $config->fill($request->all());

        $config->save();

        return redirect()
            ->back()
            ->with('message_success', 'Información actualizada');
    }

     public function showVars()
    {

        $variable = Variable::first();

        if ($variable == null) {
            $variable = new Variable();
        }

        return view('shop::variable.edit')
            ->with('variable', $variable);
    }

    public function postVars(Request $request)
    {

        $variable = Variable::first();

        if ($variable == null) {
            $variable = new Variable();
        }

        $variable->fill($request->all());

        $variable->save();

        return redirect()
            ->back()
            ->with('message_success', 'Información actualizada');
    }

}
