<?php

namespace Onestartup\Shop\DataTables;

use Onestartup\Shop\Model\ProductShop as Product;
use Yajra\DataTables\Services\DataTable;

class StockDataTable extends DataTable
{
    /**
     * Build DataTable class.
     *
     * @param mixed $query Results from query() method.
     * @return \Yajra\DataTables\DataTableAbstract
     */
    public function dataTable($query)
    {
        return datatables($query)
            ->addColumn('producto', function(Product $product) {
                    return 'Nombre: <b>'.$product->name.'</b>'.
                            '<br>Resumen: <b>'.$product->description.'</b>'.
                            '<br>Fecha de publicación: <b>'.$product->publication_date.'</b>';
                })
            ->addColumn('informacion_venta', function(Product $product) {
                    
                    $html = "";
                    $existencia = "";
                    if (isset($product->infoSale)){

                        if ($product->infoSale->quantity <= $product->infoSale->reserve_amount) {
                            $existencia = "<span class='badge danger'>".$product->infoSale->quantity."</span>";
                        } else {
                            $existencia = "<span class='badge primary'>".$product->infoSale->quantity."</span>";
                        }
                        
                        $html = "Precio venta: <b>".money_format('%(#10n', $product->infoSale->sale_price)."</b><br>Precio compra: <b>".money_format('%(#10n', $product->infoSale->purchase_price)."</b><br>Alerta de reserva: <b><span class='badge info'>".$product->infoSale->reserve_amount."</b></span><br>Existencias: $existencia<br>";
                    }else{
                        $html = "<b>Sin información de ventas</b>";
                    }
                    return $html;
                })
            ->addColumn('action',  function (Product $product){
                return "<a href='".route('admin.shop.product.edit',$product->id)."'>Editar <i class='fa fa-edit'></i></a>";
            })
            ->filterColumn('producto', function($query, $keyword) {
                $query->whereRaw("name like ?", ["%{$keyword}%"]);
            })
            ->rawColumns(['producto', 'action', 'informacion_venta']);
            
    }

    /**
     * Get query source of dataTable.
     *
     * @param \App\Interested $model
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(Product $model)
    {
        return $model->select([
            'id', 
            'name',
            'slug',
            'description',
            'active',
            'publication_date',
            'category_id'
            ])->where('active', true)->orderBy('publication_date', 'desc');
    }

    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html()
    {
        return $this->builder()
                    ->columns($this->getColumns())
                    ->parameters([
                        'dom'          => 'Bfrtip',
                        'pageLength'   => 20,
                        'buttons'      => ['excel', 'csv','reset', 'reload'],
                    ]);
    }

    /**
     * Get columns.
     *
     * @return array
     */
    protected function getColumns()
    {
        return [
            'id', 
            'producto', 
            'informacion_venta',
            'action'
        ];
    }

    /**
     * Get filename for export.
     *
     * @return string
     */
    protected function filename()
    {
        return 'orders' . date('YmdHis');
    }
}
